/*global define */
/*jshint esversion: 6 */

define(["lib/Zoot"],
function (Z) {
	"use strict";

	var triggerInputId = "TriggerInput";
	
	// Iterate over the puppets and construct a list of triggers we can use for publishing and triggering.
	function constructTriggerList (self, args) {
		var puppetTriggerSet = new Set(),
			triggerLookup = {}; // Used within the scope of this function to 

		// Keeps a list of all our triggers as a map using the groupID as the key.
		self.triggerList = {};
		
		function addTriggers(inPuppet) {
			var puppetPPath = inPuppet.getPPath();

			// Don't stick this puppet0 in the set.  It's a wrapper object and it won't be unique!  -jacquave
			if (inPuppet && !puppetTriggerSet.has(puppetPPath)) {
				puppetTriggerSet.add(puppetPPath); // Avoids iterating over the same puppet twice.
				//console.log("Set: puppetTriggerSet.has(puppet0) = " + puppetTriggerSet.has(puppetPPath));

				var triggerGroupsA0 = inPuppet.getTriggerGroups();
				if (triggerGroupsA0) {
					for (var triggerGroupIdx in triggerGroupsA0) {
						if (triggerGroupsA0.hasOwnProperty(triggerGroupIdx)) {
							var tg = triggerGroupsA0[triggerGroupIdx];
							
							self.triggerList[tg.groupID] = {triggersA:[], defaultTriggerObj0:null, puppetPPath:puppetPPath, 
															priority:tg.priority, bAllowMultipleLatches:tg.bAllowMultipleLatches}; // create an array of triggers for this group.

							for (var triggerIdx in tg.aTriggers) {
								if (tg.aTriggers.hasOwnProperty(triggerIdx)) {
									var trigger = tg.aTriggers[triggerIdx], triggerId = trigger.id, triggerObj;

									triggerObj = {triggerData:trigger, layersA:[]};	// The trigger and layers affected by it.
									self.triggerList[tg.groupID].triggersA.push(triggerObj);
									triggerLookup[triggerId] = triggerObj;

									if (tg.defaultTriggerID === triggerId) {
										//console.logToUser(`assigning default trigger ID ${triggerId}`);
										self.triggerList[tg.groupID].defaultTriggerObj0 = triggerObj;
									}
								}
							}
						}
					}
				}
			}
		}
	
		var puppet0 = args.stageLayer.getPuppet();

		if (puppet0) {
			addTriggers(puppet0);
		}
		
		args.forEachLayerInTree(function (lay) {
			var triggeredByMap = lay.getTriggeredByMap();

			for (var triggeredByMapId in triggeredByMap) {
				if (triggeredByMap.hasOwnProperty(triggeredByMapId)) {
					var trigger0 = triggerLookup[triggeredByMapId];
					if (trigger0) {
						// Remember the initial visibility (no longer used)
						var visibleB = lay.getVisible(),
							layerObject = {layer:lay, initiallyVisibleB:visibleB}; 
						
						// Add to the array of layers for this trigger.
						trigger0.layersA.push(layerObject);
						
						lay.setTriggerable(lay.getHideOthersWhenTriggered());
					}
				}
			}
		});
	}

	function getKeyGraphId (inKeyName) {
		return Z.keyCodes.getKeyGraphId(Z.keyCodes.getKeyCode(inKeyName));
	}

	function isLatched (latchValue) {
		return latchValue === 2;
	}

	function updateKeyState (args, inTriggerObj, inGraphId) {
		// Get the max value from the event graph directly, since it's live.  -jacquave
		var keyState = inTriggerObj.keyState, maxKeyValue0 = args.eventGraph.getMaxValue(inGraphId),
			lastGraphTime0 = args.eventGraph.getGraphLastTime(inGraphId),
			keyDown = args.getParamEventValue(triggerInputId, inGraphId) || 0;

		keyState.lastMaxValueMap = keyState.lastMaxValueMap || {};

		// To trigger quick keydowns that happen between frames, we track the last max value
		// for the key and enable the trigger for the next frame.  -jacquave
		if (keyState.lastMaxValueMap[inGraphId] === undefined) {
			// In order to deal with resetting the rehearsal state, we set this
			// to be valid, then can use the comparison with the last max value correctly below.  -jacquave
			keyState.lastMaxValueMap[inGraphId] = keyDown ? 0 : (maxKeyValue0 || 0);
		}
		var maxValueChangedB = (maxKeyValue0 > keyState.lastMaxValueMap[inGraphId]),
			springValueB = (!!keyDown || maxValueChangedB) || false,
			latchedValueB = keyState.wasLatchedB;

		if (maxValueChangedB) {
			// latch state changed.
			latchedValueB = !latchedValueB;
		}

		if (keyState.lastGraphTime0 === undefined || keyState.lastGraphTime0 < lastGraphTime0) {
			keyState.lastGraphTime0 = lastGraphTime0;
		}
		keyState.wasSpringedB = keyState.wasSpringedB || springValueB;
		keyState.wasLatchedB = latchedValueB;
		keyState.lastMaxValueMap[inGraphId] = maxKeyValue0 || 0;
		keyState.triggeredB = isLatched(inTriggerObj.triggerData._triggerLatch) ? latchedValueB : keyState.wasSpringedB;
	}
	
	function publishTriggers (self, args) {
		for (var triggerGroupId in self.triggerList) {
			if (self.triggerList.hasOwnProperty(triggerGroupId)) {
				var triggersA = self.triggerList[triggerGroupId].triggersA,
					triggerGroupObj = self.triggerList[triggerGroupId],
					prefixKey = args.getParamEventOutputKey(triggerInputId),
					takeTakeGroupName = args.getParamEventOutputForTriggerGroup(triggerGroupId), 
					takeTriggerKeyName = prefixKey + takeTakeGroupName,
					triggerGroupKey = "Trigger" + triggerGroupObj.puppetPPath,
					activeTrigger0 = null;
			
				// Iterate over the input (key&midi) to update each trigger's value
				for (var triggerIndex = triggersA.length-1; triggerIndex >= 0; triggerIndex--) {
					var triggerObj = triggersA[triggerIndex], keyName = triggerObj.triggerData._triggerKey,
						keyGraphId = getKeyGraphId(keyName), /*keyDown,*/
						triggerKey = triggerGroupKey + "/" + triggerObj.triggerData.id,
						midiNote = triggerObj.triggerData._nMidiNote,
						midiNoteKey, midiChannel;
					triggerObj.keyState = triggerObj.keyState || {};
					triggerObj.keyState.wasSpringedB = false;

					// Control Panel Button pushes
					updateKeyState(args, triggerObj, triggerKey + "/Down");

					// Midi
					if (midiNote) {
						midiChannel = triggerObj.triggerData._nMidiChannel;
						midiNoteKey = "Midi/" + midiChannel + "/Note/" + midiNote + "/Down";
						
//						var midiValue = args.getParamEventValue(triggerInputId, midiNoteKey) || 0; 
//						
//						if (midiValue) {
//							console.log("midiNoteKey = " + midiNoteKey + ", midiValue = " + midiValue + " (" + triggerObj.triggerData.id + ")");
//						}
						updateKeyState(args, triggerObj, midiNoteKey);
					}

					// Keyboard
					updateKeyState(args, triggerObj, keyGraphId);
					
					if (triggerObj.keyState.triggeredB && 
						(!activeTrigger0 || 
						 	(triggerGroupObj.priority !== "MostRecent" || activeTrigger0.lastGraphTime < triggerObj.keyState.lastGraphTime0))) {
						activeTrigger0 = {triggerIndex:triggerIndex, lastGraphTime:triggerObj.keyState.lastGraphTime0};
					}
                    
//					console.log("\tTrigger: " + triggerObj.triggerData.id + 
//									  ", keyGraph = " + keyName +
//									  ", keyDown = " + keyDown + 
//									  ", triggeredB = " + triggerObj.keyState.triggeredB);
				}
                
//				console.log("Trigger Group: " + triggerGroupId +
//								  ", active Trigger: " + (activeTrigger0 ? triggersA[activeTrigger0.triggerIndex].triggerData.id : "undefined"));

				if (activeTrigger0) {
					var activeTriggerObj = triggersA[activeTrigger0.triggerIndex];

					if (!triggerGroupObj.bAllowMultipleLatches && isLatched(activeTriggerObj.triggerData._triggerLatch)) {
						// Iterate over triggers to unlatch those actively latched.
						for (triggerIndex = 0; triggerIndex < triggersA.length; triggerIndex++) {
							if (activeTrigger0.triggerIndex !== triggerIndex && triggersA.hasOwnProperty(triggerIndex)) {
								var otherTriggerObj = triggersA[triggerIndex];

								if (isLatched(otherTriggerObj.triggerData._triggerLatch) && otherTriggerObj.keyState.triggeredB) {
									// Clear the latch state.
									delete otherTriggerObj.keyState;
//									console.log("\tUnlatched Trigger: " + otherTriggerObj.triggerId);
								}
							}
						}
					}

//					console.log("Trigger Group: " + triggerGroupId +
//									  ", takeTriggerKeyName: " + takeTriggerKeyName +
//									  ", takeTakeGroupName: " + takeTakeGroupName +
//									  ", active Trigger value: " + activeTrigger0.triggerIndex);

					// Publish the active trigger
					args.setEventGraphParamRecordingValid(triggerInputId, takeTakeGroupName);
					
					args.eventGraph.publish1D(takeTriggerKeyName, args.currentTime, activeTriggerObj.triggerData.intId, true);
				} else {
					if (args.eventGraph.hasGraph(takeTriggerKeyName)) {
						// If this was published before, publish it again with the non-triggered value of zero.
						args.eventGraph.publish1D(takeTriggerKeyName, args.currentTime, 0, true);			
					}
				}
			}
		}
	}

	function clearOutTriggeredByInfo (triggerObj) {
		
		triggerObj.layersA.forEach(function (inLayer) {
			// TODO: should this be done by premediate?
			delete inLayer.layer.triggeredByObjSet0;
		});
	}

	function triggerReplacements (self, triggerObj, triggerPriority) {
		
		triggerObj.layersA.forEach( function (inLayer) {
			var layer = inLayer.layer;
            
            // old way
			//console.logToUser(`triggering ${layer.getName()} with priority ${triggerPriority}`);
			layer.trigger(triggerPriority); // show this layer, and hide siblings if (old-school) mutex
            
            // new way, TODO: make real API in engine that all behaviors can call
            layer.triggeredByObjSet0 = layer.triggeredByObjSet0 || new Set();
            layer.triggeredByObjSet0.add(triggerObj);
            //console.logToUser(`layer.triggeredByObjSet0.size = ${layer.triggeredByObjSet0.size}`);
			//console.log("triggering " + layer.getName());
		});
	}
	
	var usualTriggerPriority = 1.01;	// higher than old Keyboard Triggers
	
	function chooseTriggerReplacements (self, args) {
		var inputParamId = triggerInputId;
		
		for (var triggerGroupId in self.triggerList) {
			if (self.triggerList.hasOwnProperty(triggerGroupId)) {
				var triggerGroupObj = self.triggerList[triggerGroupId],
					triggersA = triggerGroupObj.triggersA,
					prefixKey = args.getParamEventOutputKey(triggerInputId),
					takeTakeGroupName = args.getParamEventOutputForTriggerGroup(triggerGroupId), 
					takeTriggerKeyName = prefixKey + takeTakeGroupName, 
					activeValue;
			
				activeValue = args.getParamEventValue(inputParamId, takeTriggerKeyName, takeTakeGroupName, undefined, true);
				
//				console.log("Trigger Group: " + triggerGroupId +
//								  ", takeTriggerKeyName: " + takeTriggerKeyName +
//								 ", active Trigger value: " + activeValue);
				
					var triggerObj, triggerIndex,
						triggerPriority = usualTriggerPriority,
						retriggerPriority,
						retriggeredTriggerObj0 = null,
						activeTriggerObj0 = null;

				// could avoid searching here by stuffing group & obj directly in retriggerOnNextFrame,
				//	but we still need to iterate over all clears to clear things out, so wouldn't help much
                    for (triggerIndex = 0; triggerIndex < triggersA.length; triggerIndex++) {
                        triggerObj = triggersA[triggerIndex];

                        if (triggerObj.bRetrigger) {
                            triggerObj.bRetrigger = false;    // clear out for next frame
							retriggerPriority = triggerObj.retriggerPriority;
                            Z.utils.assert(!retriggeredTriggerObj0, "found more than one re-triggered trigger");
							retriggeredTriggerObj0 = triggerObj;
							// trigger was retriggered, but may also happen to be directly triggered;
							//	in this case if someone asks if it was retriggered we say no -- as what
							//	they're really asking is it _only_ retriggered and not triggered the normal way

							// no break -- loop over all to clear out bWasRetriggered
                        }
                        triggerObj.bWasRetriggered = false;
						clearOutTriggeredByInfo(triggerObj);
                    }
                        
                    if (activeValue) {
						// Find the active value.
						for (triggerIndex = 0; triggerIndex < triggersA.length; triggerIndex++) {
							triggerObj = triggersA[triggerIndex];
							
							if (triggerObj.triggerData.intId === activeValue) {
								activeTriggerObj0 = triggerObj;
								break;
							}
						}
					}
					if (!activeTriggerObj0) {
						if (triggerGroupObj.defaultTriggerObj0) {
							activeTriggerObj0 = triggerGroupObj.defaultTriggerObj0;
							triggerPriority = -10;	// allow any other behavior to trump the default (e.g. Head Turner)
						}
					}
					
					// re-triggered trumps others
					var finalTriggerObj0 = retriggeredTriggerObj0 || activeTriggerObj0;
					
					if (finalTriggerObj0) {
						 // so getTriggeringType can tell if it was solely because of a retrigger
						if (retriggeredTriggerObj0 && (retriggeredTriggerObj0 !== activeTriggerObj0)) {
							finalTriggerObj0.bWasRetriggered = true;
							triggerPriority = retriggerPriority;	// match original priority
						}
						
						//console.logToUser(`finalTriggerObj0 = ${finalTriggerObj0.triggerData.id}, activeTriggerObj0 = ${activeTriggerObj0.triggerData.id}, finalTriggerObj0.bWasRetriggered = ${finalTriggerObj0.bWasRetriggered}`);
						
						triggerReplacements(self, finalTriggerObj0, triggerPriority);
						
					}
			}
		}
	}	

	return {
		about:			"$$$/private/animal/Behavior/Triggers/About=Triggers, (c) 2014.",
		description: 	"$$$/animal/Behavior/Triggers/Desc=Hides and shows artwork based on triggers",
		uiName:  		"$$$/animal/Behavior/Triggers/UIName=Triggers",
		defaultArmedForRecordOn: true,

		defineParams: function () { // free function, called once ever; returns parameter definition (hierarchical) array
			return [
				{
				    id: triggerInputId, 
					type: "eventGraph",
					uiName: "$$$/animal/Behavior/Triggers/Parameter/TriggerInput=Trigger Input", 
					inputKeysArray: ["Keyboard/", "Midi/", "Trigger/"],
			        outputKeyTraits: {
		                takeGroupsArray: [
			                {
			                    id: "TriggerGroup/*",
								uiName : "$$$/private/animal/Behavior/Triggers/takeGroupName={triggerGroupName:*}"
							}
		                ],
						editableB:true, // Not yet ready!!
						takeData : {	// This may become an array of objects at some point, since a take can have multiple data streams.
							eventGraphKey : "{triggerGroupId:*}",
							type : "triggers",
							redundancyTraits : {v : 0, bNilKeyframeIsRedundant : true},
							uiLabel : "",
							uiName : "$$$/animal/Behavior/Triggers/Value/TriggersName=Trigger",
							uiNamePlural : "$$$/animal/Behavior/Triggers/Value/TriggersNamePlural=Triggers"
						},
					},
				    uiToolTip: "$$$/animal/Behavior/Triggers/Parameter/TriggerInput/tooltip=Keyboard input used to hide and show artwork", defaultArmedForRecordOn: true
				}
			];
		},

		onCreateBackStageBehavior : function (/*self*/) {
			return { order: 0.0, importance : 0.0 };	// must come before Cycle Layers or any other behavior that reads triggers
		},

		onCreateStageBehavior : function (self, args) {
			constructTriggerList(self, args);
		},

		onResetRehearsalData : function (self) {
			for (var triggerGroupId in self.triggerList) {
				if (self.triggerList.hasOwnProperty(triggerGroupId)) {
					var triggersA = self.triggerList[triggerGroupId].triggersA;
					for (var triggerIndex = 0; triggerIndex < triggersA.length; triggerIndex++) {
						var triggerObj = triggersA[triggerIndex];
						// Remove the latch state.
						delete triggerObj.keyState;
					}
				}
			}
		},

		onFilterLiveInputs : function (self, args) { // method on behavior that is attached to a puppet, only onstage
			var inputParamId = triggerInputId, inputLiveB = args.isParamEventLive(inputParamId);
			
			if (inputLiveB) {
				publishTriggers(self, args);
			}
		},

		onAnimate : function (self, args) { // method on behavior that is attached to a puppet, only onstage	
			chooseTriggerReplacements(self, args);
		}

	}; // end of object being returned
});
